<?php

namespace App\Models;

use App\Enums\VisitorStatus;
use App\Enums\VisitorType;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Visitor extends Model
{
    use HasFactory;

    protected $attributes = [
        'status' => VisitorStatus::PENDING,
        'visitor_type' => VisitorType::GUEST,
    ];

    protected $fillable = [
        'name',
        'phone',
        'email',
        'visitor_type',
        'wing_id',
        'flat_id',
        'purpose',
        'photo',
        'status',
        'entry_time',
        'exit_time',
        'approved_by',
        'approved_at',
        'approval_notes',
        'rejection_reason',
        'checked_in_by',
        'checked_out_by',
        'additional_info',
        'is_recurring',
    ];

    protected function casts(): array
    {
        return [
            'visitor_type' => VisitorType::class,
            'status' => VisitorStatus::class,
            'entry_time' => 'datetime',
            'exit_time' => 'datetime',
            'approved_at' => 'datetime',
            'additional_info' => 'array',
            'is_recurring' => 'boolean',
        ];
    }

    public function wing(): BelongsTo
    {
        return $this->belongsTo(Wing::class);
    }

    public function flat(): BelongsTo
    {
        return $this->belongsTo(Flat::class);
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function checkedInBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'checked_in_by');
    }

    public function checkedOutBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'checked_out_by');
    }

    public function isPending(): bool
    {
        return $this->status === VisitorStatus::PENDING;
    }

    public function isApproved(): bool
    {
        return $this->status === VisitorStatus::APPROVED;
    }

    public function isRejected(): bool
    {
        return $this->status === VisitorStatus::REJECTED;
    }

    public function isCheckedIn(): bool
    {
        return $this->status === VisitorStatus::CHECKED_IN;
    }

    public function isCheckedOut(): bool
    {
        return $this->status === VisitorStatus::CHECKED_OUT;
    }

    public function isExpired(): bool
    {
        return $this->status === VisitorStatus::EXPIRED;
    }

    public function canCheckOut(): bool
    {
        return $this->isCheckedIn();
    }
}
