<?php

namespace App\Http\Controllers\Api;

use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\Vehicle\StoreVehicleRequest;
use App\Http\Requests\Vehicle\UpdateVehicleRequest;
use App\Http\Resources\VehicleResource;
use App\Models\Vehicle;
use App\Traits\CanManageFile;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Spatie\QueryBuilder\QueryBuilder;

class VehicleController extends Controller
{
    use CanManageFile;

    public function __construct() {
        $this->resourceName = "vehicles";
        $this->fileFieldNames = ["photo", "insurance_doc"];
        $this->fileFolderPaths = ["photos", "documents"];
    }

    public function index() {
        $query = QueryBuilder::for(Vehicle::class)
            ->allowedFilters(['owner_id', 'flat_id', 'vehicle_type', 'make', 'model', 'registration_no', 'is_active'])
            ->allowedIncludes(['owner', 'flat'])
            ->latest();

        $vehicles = $query
            ->paginate(request("perPage", 15));

        return ApiResponse::paginated($vehicles);
    }

    public function store(StoreVehicleRequest $request) {
        $vehicle = new Vehicle($request->validated());
        $vehicle->unique_code = $this->generateUniqueCode();
        $this->storeFiles($request, $vehicle);
        $vehicle->save();

        return ApiResponse::created(
            new VehicleResource($vehicle->load(['owner', 'flat'])),
            "Vehicle has been created successfully!"
        );
    }

    public function show(Vehicle $vehicle) {
        return ApiResponse::resource(
            new VehicleResource($vehicle->load(['owner', 'flat']))
        );
    }

    public function update(UpdateVehicleRequest $request, Vehicle $vehicle) {
        $vehicle->fill($request->validated());
        $this->updateFiles($request, $vehicle);
        $vehicle->save();

        return ApiResponse::updated(
            new VehicleResource($vehicle->load(['owner', 'flat'])),
            "Vehicle has been updated successfully!"
        );
    }

    public function destroy(Vehicle $vehicle) {
        $this->deleteFiles($vehicle);
        $vehicle->delete();

        return ApiResponse::deleted("Vehicle has been deleted successfully!");
    }

    private function generateUniqueCode(): string {
        do {
            $code = 'VEH' . strtoupper(Str::random(8));
        } while (Vehicle::where('unique_code', $code)->exists());

        return $code;
    }
}
