<?php

namespace App\Helpers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Http\Resources\Json\ResourceCollection;
use Illuminate\Pagination\LengthAwarePaginator;
use Symfony\Component\HttpFoundation\Response;

class ApiResponse
{
    public static function validationError($errors, string $message = 'Validation failed'): JsonResponse {
        return self::error($message, Response::HTTP_UNPROCESSABLE_ENTITY, $errors);
    }

    /**
     * General error response
     */
    public static function error(
        string $message = 'Error',
        int $statusCode = Response::HTTP_BAD_REQUEST,
        $errors = null,
        $data = null
    ): JsonResponse {
        $response = [
            'success' => false,
            'message' => $message,
        ];

        if (!is_null($errors)) {
            $response['errors'] = $errors;
        }

        if (!is_null($data)) {
            $response['data'] = $data;
        }

        return response()->json($response, $statusCode);
    }

    public static function unauthorized(string $message = 'Unauthorized'): JsonResponse {
        return self::error($message, Response::HTTP_UNAUTHORIZED);
    }

    public static function forbidden(string $message = 'Forbidden'): JsonResponse {
        return self::error($message, Response::HTTP_FORBIDDEN);
    }

    public static function notFound(string $message = 'Resource not found'): JsonResponse {
        return self::error($message, Response::HTTP_NOT_FOUND);
    }

    public static function serverError(string $message = 'Internal server error'): JsonResponse {
        return self::error($message, Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public static function created($data = null, string $message = 'Resource created successfully'): JsonResponse {
        return self::success($data, $message, Response::HTTP_CREATED);
    }

    /**
     * General success response
     */
    public static function success(
        $data = null,
        string $message = 'Success',
        int $statusCode = Response::HTTP_OK,
        array $meta = []
    ): JsonResponse {
        $response = [
            'success' => true,
            'message' => $message,
        ];

        if (!is_null($data)) {
            $response['data'] = $data;
        }

        if (!empty($meta)) {
            $response['meta'] = $meta;
        }

        return response()->json($response, $statusCode);
    }

    public static function updated($data = null, string $message = 'Resource updated successfully'): JsonResponse {
        return self::success($data, $message);
    }

    public static function deleted(string $message = 'Resource deleted successfully'): JsonResponse {
        return self::success(null, $message);
    }

    public static function noContent(): JsonResponse {
        return response()->json(null, Response::HTTP_NO_CONTENT);
    }

    public static function paginated(
        LengthAwarePaginator $paginator,
        string $message = 'Data retrieved successfully'
    ): JsonResponse {
        return self::success(
            $paginator->items(),
            $message,
            Response::HTTP_OK,
            [
                'current_page' => $paginator->currentPage(),
                'last_page' => $paginator->lastPage(),
                'per_page' => $paginator->perPage(),
                'total' => $paginator->total(),
                'from' => $paginator->firstItem(),
                'to' => $paginator->lastItem(),
                'has_more_pages' => $paginator->hasMorePages(),
            ]
        );
    }

    public static function resource(
        JsonResource $resource,
        string $message = 'Data retrieved successfully'
    ): JsonResponse {
        return self::success($resource, $message);
    }

    public static function collection(
        ResourceCollection $collection,
        string $message = 'Data retrieved successfully'
    ): JsonResponse {
        return self::success($collection, $message);
    }
}
